/**
 * \file: Sensor.cpp
 *
 * \version: $Id:$
 *
 * \release: $Name:$
 *
 * <brief description>.
 * <detailed description>
 * \component: Android Auto
 *
 * \author: M. Adachi / ADIT/SW / madachi@jp.adit-jv.com
 *
 * \copyright (c) 2014 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 * \see <related items>
 *
 * \history
 *
 ***********************************************************************/
#include <adit_logging.h>
#include "Sensors.h"
#include "SensorLocationData.h"

LOG_IMPORT_CONTEXT(aauto_sensor)

namespace adit { namespace aauto {

using namespace std;

//------------------------------------------------------------------
// Sensor test class
//------------------------------------------------------------------
SensorTest::SensorTest(bool inVerbose, sensorTable& inSensor) : verbose(inVerbose), sensor(inSensor)
{
    remainingTime = 0;
    setMode(aautoSensorWork);               // test mode is fixed count loop

    testCount = sensor.testCount;           // set count for test (test mode is fixed count loop) 
    if(testCount == 0)
    {
        setMode(aautoSensorEndlessWork);    // test mode is endless loop
    }
    else if(testCount == -1)
    {
        setMode(aautoSensorBreak);          // test mode is dynamic start
    }
}

void SensorTest::setMode(sensorTestMode mode) 
{
    Autolock m(&mLock);

    testMode = mode;
}

int SensorTest::start(::shared_ptr<SensorSource> inSensorSource)
{
    int err_code = STATUS_SUCCESS;

    sensorSource = inSensorSource;
    if(remainingTime == 0)
    {
        if(testMode != aautoSensorStop)
        {
            err_code = ExecTest(testMode);      // execution test
            remainingTime = sensor.interval_ms;
            if(testMode == aautoSensorWork)
            {
                if(testCount-- == 0)
                {
                    setMode(aautoSensorStop);
                }
            }
        }
    }
    else
    {
        remainingTime--;
    }

    return err_code;
}

//------------------------------------------------------------------
// Night Mode class
//------------------------------------------------------------------
NightModeData::NightModeData(bool inVerbose, sensorTable& inSensor) : SensorTest(inVerbose, inSensor)
{
    nightModeFlag = false;
}

int NightModeData::ExecTest(sensorTestMode& testMode)
{
    nightModeFlag = !nightModeFlag;
    if(verbose)
    {
        LOGD_DEBUG((aauto_sensor, "Night Mode enable (%d)", nightModeFlag));
    }

    int err_code = sensorSource->reportNightModeData(nightModeFlag);
    if(err_code != STATUS_SUCCESS)
    {
        LOG_ERROR((aauto_sensor, "failed report NightMode data"));
    }

    return err_code;
}

//------------------------------------------------------------------
// Speed data class
//------------------------------------------------------------------
SpeedData::SpeedData(bool inVerbose, sensorTable& inSensor) : SensorTest(inVerbose, inSensor)
{
    speedDataCount = 0;
    hasCruiseEngaged = false;
    cruiseEngaged = false;
    hasCruiseSetSpeed = false;
    cruiseSetSpeed = 0;
}

int SpeedData::ExecTest(sensorTestMode& testMode)
{
    if(verbose)
    {
        LOGD_DEBUG((aauto_sensor, "Speed Data = %d m/s", speedDataCount));
    }

    int err_code = sensorSource->reportSpeedData(speedDataCount, hasCruiseEngaged, cruiseEngaged,
                                                 hasCruiseSetSpeed, cruiseSetSpeed);

    if(err_code == STATUS_SUCCESS)
    {
        speedDataCount += 10;
        speedDataCount %= 10000;
    }
    else
    {
        LOG_ERROR((aauto_sensor, "failed report speed data"));
    }

    return err_code;
}

//------------------------------------------------------------------
// Compass data class
//------------------------------------------------------------------
CompassData::CompassData(bool inVerbose, sensorTable& inSensor) : SensorTest(inVerbose, inSensor)
{
    compassDataCount = 0;
    compassDataCount3D = 0;
    pitchE6 = -90;
    rollE6  = -180;
}

int CompassData::ExecTest(sensorTestMode& testMode)
{
    int err_code = STATUS_SUCCESS;

    if(verbose)
    {
        LOGD_DEBUG((aauto_sensor, "Compass Data = %d ", compassDataCount));
    }

    // Compass test
    err_code = sensorSource->reportCompassData(compassDataCount * SData_1e6);
    if(err_code == STATUS_SUCCESS)
    {
        compassDataCount += 1;
        if(compassDataCount > 360)
        {
            compassDataCount = 0;
        }
        
        // Compass 3D test
        if(verbose)
        {
            LOGD_DEBUG((aauto_sensor, "Compass Data 3D = %d ", compassDataCount3D));
        }

        err_code = sensorSource->reportCompassData3D(compassDataCount3D * SData_1e6,
                                    true, pitchE6 * SData_1e6, true, rollE6 * SData_1e6);
        if(err_code == STATUS_SUCCESS)
        {
            compassDataCount3D += 1;
            if(compassDataCount3D > 360)
            {
                compassDataCount3D = 0;
            }
            pitchE6 += 1;
            if(pitchE6 > 90)
            {
                pitchE6 = -90;
            }
            rollE6 += 1;
            if(rollE6 > 180)
            {
                rollE6 = -180;
            }
        }
        else
        {
            LOG_ERROR((aauto_sensor, "failed report compass data 3D"));
        }
    }
    else
    {
        LOG_ERROR((aauto_sensor, "failed report compass data"));
    }
    
   return err_code;
}

//------------------------------------------------------------------
// RPM data class
//------------------------------------------------------------------
RPMData::RPMData(bool inVerbose, sensorTable& inSensor) : SensorTest(inVerbose, inSensor)
{
    rpmDataCount = 0;
}

int RPMData::ExecTest(sensorTestMode& testMode)
{
    if(verbose)
    {
        LOGD_DEBUG((aauto_sensor, "RPM Data = %d m/s", rpmDataCount));
    }

    int err_code = sensorSource->reportRpmData(rpmDataCount * SData_1e3);
    if(err_code == STATUS_SUCCESS)
    {
        rpmDataCount += 1;
        if(rpmDataCount > 2000)
        {
            rpmDataCount = 0;
        }
    }
    else
    {
        LOG_ERROR((aauto_sensor, "failed report rpm data"));
    }

    return err_code;
}    

//------------------------------------------------------------------
// Fuel data class
//------------------------------------------------------------------
FuelData::FuelData(bool inVerbose, sensorTable& inSensor) : SensorTest(inVerbose, inSensor)
{
    fuelRemainingPercent = 100;
}

int FuelData::ExecTest(sensorTestMode& testMode)
{
    if(verbose)
    {
        LOGD_DEBUG((aauto_sensor, "Fuel Data = Remaining %dl", fuelRemainingPercent));
    }

    fuelRemainingPercent -= 1;
    if(fuelRemainingPercent < 0)
    {
        fuelRemainingPercent = 100;
    }

    int32_t rangeKm = fuelRemainingPercent / 5;

    bool lowFuelWarning = false;
    if(fuelRemainingPercent < 5)
    {
        lowFuelWarning = true;
    }

    int err_code = sensorSource->reportFuelData(true, fuelRemainingPercent, true, rangeKm, true, lowFuelWarning);
    if(err_code != STATUS_SUCCESS)
    {
        LOG_ERROR((aauto_sensor, "failed report fuel data"));
    }

    return err_code;
}

//------------------------------------------------------------------
// Odometer data class
//------------------------------------------------------------------
OdometerData::OdometerData(bool inVerbose, sensorTable& inSensor) : SensorTest(inVerbose, inSensor)
{
    odometerDataCount = 0;
    hasTripKms = false;
    tripKmsE1 = 0;
}

int OdometerData::ExecTest(sensorTestMode& testMode)
{
    if(verbose)
    {
        LOGD_DEBUG((aauto_sensor, "Odometer Data = %d km", odometerDataCount));
    }

    int err_code = sensorSource->reportOdometerData(odometerDataCount, hasTripKms, tripKmsE1);
    if(err_code == STATUS_SUCCESS)
    {
        odometerDataCount += 1;
        odometerDataCount %= 1000;
    }
    else
    {
        LOG_ERROR((aauto_sensor, "failed report odometer data"));
    }

    return err_code;
}

//------------------------------------------------------------------
// ParkingBrake data class
//------------------------------------------------------------------
ParkingBrakeData::ParkingBrakeData(bool inVerbose, sensorTable& inSensor) : SensorTest(inVerbose, inSensor)
{
    parkingBrakeFlag = true;
    parkingBrakeIntervalCount = 100;
}

int ParkingBrakeData::ExecTest(sensorTestMode& testMode)
{
    if(!(parkingBrakeIntervalCount--))
    {
        parkingBrakeIntervalCount = 100;
        parkingBrakeFlag = !parkingBrakeFlag; 
    }

    if(verbose)
    {
        LOGD_DEBUG((aauto_sensor, "ParkingBrake data (%d)", parkingBrakeFlag));
    }

    int err_code = sensorSource->reportParkingBrakeData(parkingBrakeFlag);
    if(err_code != STATUS_SUCCESS)
    {
        LOG_ERROR((aauto_sensor, "failed report ParkingBrake data"));
    }

    return err_code;
}

//------------------------------------------------------------------
// Gear data class
//------------------------------------------------------------------
GearData::GearData(bool inVerbose, sensorTable& inSensor) : SensorTest(inVerbose, inSensor)
{
    GearDataCount = 0;
}

int GearData::ExecTest(sensorTestMode& testMode)
{ 
    Gear GearTbl[] = {
        GEAR_NEUTRAL, GEAR_1, GEAR_2, GEAR_3, GEAR_4, GEAR_5, GEAR_6, GEAR_DRIVE, GEAR_PARK, GEAR_REVERSE
    };
    
    if(verbose)
    {
        LOGD_DEBUG((aauto_sensor, "report Gear data (%d)", GEAR_PARK));
    }

    int err_code = sensorSource->reportGearData(GEAR_PARK);
    if(err_code == STATUS_SUCCESS)
    {
        if(GearDataCount >= sizeof(GearTbl) / sizeof(GearTbl[0]))
        {
            GearDataCount = 0;
        }
    }
    else
    {
        LOG_ERROR((aauto_sensor, "failed report Gear data"));
    }

    return err_code;
}

//------------------------------------------------------------------
// Door data class
//------------------------------------------------------------------
DoorData::DoorData(bool inVerbose, sensorTable& inSensor) : SensorTest(inVerbose, inSensor)
{
    verbose = inVerbose;
    HoodOpenFlag = false;
    trunkOpenFlag = false;
    DoorDataCount = 0;
}

int DoorData::ExecTest(sensorTestMode& testMode)
{
    bool doorData[] = {false, false, false, false};
    
    if(verbose)
    {
        LOGD_DEBUG((aauto_sensor, "report Door data (%d)", DoorDataCount));
    }

    doorData[DoorDataCount] = true;

    int err_code = sensorSource->reportDoorData(true, HoodOpenFlag, true, trunkOpenFlag, 
                doorData, sizeof(doorData)/sizeof(doorData[0]));
    if(err_code == STATUS_SUCCESS)
    {
        DoorDataCount++;
        if(DoorDataCount > sizeof(doorData)/sizeof(doorData[0]))
        {
            DoorDataCount = 0;
        }
        HoodOpenFlag = !HoodOpenFlag;
        trunkOpenFlag = !trunkOpenFlag;
    }
    else
    {
        LOG_ERROR((aauto_sensor, "failed report door data"));
    }

    return err_code;
}

//------------------------------------------------------------------
// Driving Status data class
//------------------------------------------------------------------
DrivingStatusData::DrivingStatusData(bool inVerbose, sensorTable& inSensor) : SensorTest(inVerbose, inSensor)
{
    verbose = inVerbose;
    drivingStatus = DRIVE_STATUS_UNRESTRICTED;
}

int DrivingStatusData::ExecTest(sensorTestMode& testMode)
{
    if(verbose)
    {
//        LOGD_DEBUG((aauto_sensor, "report driving status data (%d)", drivingStatus));
    }

    int err_code = sensorSource->reportDrivingStatusData(drivingStatus);
    if(err_code != STATUS_SUCCESS)
    {
        LOG_ERROR((aauto_sensor, "failed report driving status data"));
    }

    return err_code;
}

//------------------------------------------------------------------
// Location data class
//------------------------------------------------------------------
// location data class for Navigation 
LocationNaviData::LocationNaviData(string inCityname, bool inVerbose)
 : cityName(inCityname), verbose(inVerbose)
{
    static map <string, map<int, sensorLocationData>> 
                                loctionTestData = { { "Hildesheim", gLocDataHildesheim  },
                                                    { "Kariya",     gLocDataKariya      }};
    if (loctionTestData.find(inCityname) != loctionTestData.end()) {
        locationData = loctionTestData[inCityname];
    }
    else
    {
        cityName = "";
    }
    
    locCount = 0;
}

bool LocationNaviData::isLocationDataEmpty()
{
    bool ret = false;

    if(locCount + 1 == locationData.size())
    {
        ret = true;
    }
    
    return ret;
}

sensorLocationData LocationNaviData::getLocationData()
{
    return locationData[locCount];
}

void LocationNaviData::updatePosition()
{
    if(posCount++ < locationData[locCount].incDecCount)
    {
        latitude += incDecLat;
        longitude += incDecLong;
    }
    else
    {
        locCount++;
        updateLocation();
    }
}
void LocationNaviData::initilizeLocation()
{
    locCount = 0;
    updateLocation();
}

void LocationNaviData::updateLocation()
{
    int32_t deltaPos = 0;

    if(locCount + 1 < locationData.size())
    {
    // position initialize
        posCount = 0;
         
    // latitude calculate
        latitude = locationData[locCount].position.latitudeE7;
        deltaPos = locationData[locCount + 1].position.latitudeE7 - latitude;
        incDecLat = deltaPos / locationData[locCount].incDecCount;
        
    // longitude calculate
        longitude = locationData[locCount].position.longitudeE7;
        deltaPos = locationData[locCount + 1].position.longitudeE7 - longitude;
        incDecLong = deltaPos / locationData[locCount].incDecCount;
    }

    if(verbose)
    {
        LOGD_DEBUG((aauto_sensor, "---> Update location data incDecLat = %d incDecLong = %d locCount = %d",
         incDecLat, incDecLong, locCount));
    }
}

// location data handle class
LocationData::LocationData(bool inVerbose, sensorTable& inSensor, string inLocationStr) : SensorTest(inVerbose, inSensor)
{
    verbose = inVerbose;
    locData = new LocationNaviData(inLocationStr, verbose);
    locData->initilizeLocation();
}

int LocationData::ExecTest(sensorTestMode& testMode)
{
    struct timespec tp;
    clock_gettime(CLOCK_MONOTONIC, &tp);
    uint64_t timestamp = (tp.tv_sec * NSEC_PER_SEC + tp.tv_nsec) / 1000;

    int err_code = STATUS_SUCCESS;

    if(locData->getLocName() == "") // check registered location data table
    {
        LOGD_DEBUG((aauto_sensor, "location data does not work because there is not location table."));
        testMode = aautoSensorStop; // test stop
        return err_code;
    }

    if(verbose)
    {
        LOGD_DEBUG((aauto_sensor, "report location data latitude x longitude: (%d x %d) locCount: %d testMode = %d",
                                        locData->getLatitude(), locData->getLongitude(), locData->getLocCount(), testMode));
    }

    if(testMode != aautoSensorStop)
    {
        err_code = sensorSource->reportLocationData(    timestamp,
                                                        locData->getLatitude(),
                                                        locData->getLongitude(),
                                                        true,
                                                        locData->getLocationData().accuracyE3,
                                                        true,
                                                        locData->getLocationData().altitudeE2,
                                                        true,
                                                        locData->getLocationData().speedE3,
                                                        true,
                                                        locData->getLocationData().bearingE6 
                                                    );

        if(testMode != aautoSensorBreak)
        {
            locData->updatePosition();
        }

        if(locData->isLocationDataEmpty())
        {
            setMode(aautoSensorStop);    // stop test
            locData->initilizeLocation();
        }

        if(err_code != STATUS_SUCCESS)
        {
            LOG_ERROR((aauto_sensor, "failed report location data"));
        }
    }

    return err_code;
}

} } // namespace adit { namespace aauto {
